<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\UserVipInvestment;
use App\Models\VipProduct;
use App\Models\Transaction;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class VipInvestmentController extends Controller
{
    public function invest(VipProduct $product)
    {
        return DB::transaction(function () use ($product) {
            $user = Auth::user();
            
            // লগ যোগ করুন
            Log::info('Investment attempt', [
                'user_id' => $user->id,
                'product_id' => $product->id
            ]);

            // চেক করুন ব্যবহারকারী এই পণ্য আগে কিনেছে কিনা (যেকোনো সময়)
            if ($user->hasActiveInvestmentIn($product->id)) {
                Log::warning('User already purchased this product', [
                    'user_id' => $user->id,
                    'product_id' => $product->id
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'আপনি এই ভিআইপি পণ্যটি ইতিমধ্যে কিনেছেন! প্রতিটি পণ্য শুধুমাত্র একবারই কেনা সম্ভব।',
                    'already_purchased' => true
                ], 400);
            }

            // ব্যালেন্স চেক করুন
            if ($user->balance < $product->price) {
                return response()->json([
                    'success' => false,
                    'message' => 'আপনার ব্যালেন্স অপর্যাপ্ত!',
                    'balance_shortage' => true
                ], 400);
            }

            // এক্সপায়ার ডেট সেট করুন
            $expiresAt = now()->addDays($product->duration_days);

            // বিনিয়োগ তৈরি করুন
            $investment = UserVipInvestment::create([
                'user_id' => $user->id,
                'vip_product_id' => $product->id,
                'amount' => $product->price,
                'daily_earnings' => $product->earnings_per_user,
                'expires_at' => $expiresAt,
                'is_active' => true
            ]);

            // ব্যালেন্স কমানো
            $user->decrement('balance', $product->price);

            // পণ্যের ইনভেস্টেড ইউজার বাড়ানো (লক ব্যবহার করে)
            DB::table('vip_products')
                ->where('id', $product->id)
                ->lockForUpdate()
                ->increment('invested_users');

            // ট্রানজেকশন রেকর্ড করুন
            Transaction::create([
                'user_id' => $user->id,
                'type' => 'vip_investment',
                'amount' => $product->price,
                'status' => 'approved',
                'notes' => $product->investment_tool.' পণ্যে বিনিয়োগ',
                'transaction_id' => 'VIP-INV-'.uniqid()
            ]);

            Log::info('Investment successful', [
                'investment_id' => $investment->id,
                'user_id' => $user->id
            ]);

            return response()->json([
                'success' => true,
                'message' => 'বিনিয়োগ সফল হয়েছে!',
                'investment' => [
                    'expires_at' => $investment->expires_at->format('d/m/Y')
                ],
                'balance' => $user->fresh()->balance
            ]);
        });
    }

   // App\Http\Controllers\Frontend\VipInvestmentController.php

public function processDailyEarnings()
{
    // শুধুমাত্র active ইনভেস্টমেন্ট প্রসেস করবে (expires_at এখনকার সময়ের চেয়ে বেশি)
    $investments = UserVipInvestment::active()
        ->where(function($query) {
            $query->whereNull('last_payout_at')
                  ->orWhere('last_payout_at', '<', Carbon::now()->subDay());
        })
        ->with(['user', 'product'])
        ->get();

    $processedCount = 0;

    foreach ($investments as $investment) {
        DB::beginTransaction();
        
        try {
            // দৈনিক আয় যোগ করুন
            $investment->user->increment('balance', $investment->daily_earnings);
            
            // ট্রানজেকশন রেকর্ড করুন
            Transaction::create([
                'user_id' => $investment->user_id,
                'type' => 'vip_earnings',
                'amount' => $investment->daily_earnings,
                'method' => 'vip_investment',
                'status' => 'approved',
                'transaction_id' => 'VIP-EARN-'.uniqid(),
                'notes' => "Daily earnings from VIP product: {$investment->product->investment_tool}"
            ]);
            
            // শেষ পেমেন্টের সময় আপডেট করুন
            $investment->update(['last_payout_at' => Carbon::now()]);
            
            $processedCount++;
            DB::commit();
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Failed to process VIP earnings for investment {$investment->id}: " . $e->getMessage());
        }
    }

    return response()->json([
        'success' => true,
        'processed' => $processedCount,
        'message' => "Processed {$processedCount} active VIP investments"
    ]);
}
}