<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Transaction;
use App\Models\UserVipInvestment;
use App\Models\VipProduct;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;

class DashboardController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        
        // Apple Analysis Data
        $appleAnalysis = $this->getAppleAnalysis($user);
        $appleChange = $this->getAppleChange($user);
        
        // Cash In Data
        $cashIn = $this->getCashIn($user);
        $cashInChange = $this->getCashInChange($user);
        
        // Cashback Data
        $cashback = $this->getCashback($user);
        $cashbackChange = $this->getCashbackChange($user);
        
        // Popular Advice (VIP Product)
        $popularAdvice = $this->getPopularAdvice($user);
        
        // Risk Avoidance
        $riskAvoidance = $this->getRiskAvoidance($user);

          $withdrawTotal = \App\Models\Withdrawal::where('user_id', $user->id)
        ->where('status', 'approved') // Approved Withdrawals only
        ->sum('amount');
        
        return view('dashboard', compact(
            'appleAnalysis',
            'appleChange',
            'cashIn',
            'cashInChange',
            'cashback',
            'cashbackChange',
            'popularAdvice',
            'riskAvoidance',
            'withdrawTotal',
        ));
    }

    public function riskCheck()
{
    $user = auth()->user();

    // User-এর ইনভেস্টমেন্ট ডাইভার্সিটি ডেটা
    $investmentTypes = \App\Models\UserVipInvestment::where('user_id', $user->id)
        ->select('type')
        ->distinct()
        ->count();

    $totalInvestments = \App\Models\UserVipInvestment::where('user_id', $user->id)->count();

    $riskAvoidance = $this->getRiskAvoidance($user);

    // ডাইনামিক টিপস তৈরি
    $tips = [];
    if ($investmentTypes <= 1) {
        $tips[] = "আপনার সমস্ত বিনিয়োগ একই ধরনের। বৈচিত্র আনুন ঝুঁকি কমাতে।";
    }
    if ($totalInvestments < 3) {
        $tips[] = "আরও কিছু ইনভেস্টমেন্ট অপশন ব্যবহার করুন নিরাপত্তা বাড়াতে।";
    }
    if (empty($tips)) {
        $tips[] = "দারুণ! আপনার ইনভেস্টমেন্ট বেশ বৈচিত্রময়।";
    }

    return view('risk-check', compact('riskAvoidance', 'tips', 'investmentTypes', 'totalInvestments'));
}


    

    // Helper methods to get data
    private function getAppleAnalysis($user)
    {
        return UserVipInvestment::where('user_id', $user->id)
            ->where('type', 'apple')
            ->sum('amount');
    }

    private function getAppleChange($user)
    {
        $current = $this->getAppleAnalysis($user);
        $previous = UserVipInvestment::where('user_id', $user->id)
            ->where('type', 'apple')
            ->whereDate('created_at', '<', now()->subMonth())
            ->sum('amount');
            
        return $previous ? (($current - $previous) / $previous) * 100 : 0;
    }

    private function getCashIn($user)
    {
        return Transaction::where('user_id', $user->id)
            ->where('type', 'deposit')
            ->where('status', 'approved')
            ->sum('amount');
    }

    private function getCashInChange($user)
    {
        $currentMonth = $this->getCashIn($user);
        $previousMonth = Transaction::where('user_id', $user->id)
            ->where('type', 'deposit')
            ->where('status', 'approved')
            ->whereMonth('created_at', now()->subMonth()->month)
            ->sum('amount');
            
        return $previousMonth ? (($currentMonth - $previousMonth) / $previousMonth) * 100 : 0;
    }

    private function getCashback($user)
    {
        return Transaction::where('user_id', $user->id)
            ->where('type', 'cashback')
            ->sum('amount');
    }

    private function getCashbackChange($user)
    {
        $currentMonth = $this->getCashback($user);
        $previousMonth = Transaction::where('user_id', $user->id)
            ->where('type', 'cashback')
            ->whereMonth('created_at', now()->subMonth()->month)
            ->sum('amount');
            
        return $previousMonth ? (($currentMonth - $previousMonth) / $previousMonth) * 100 : 0;
    }

  private function getPopularAdvice($user)
{
    $popularProduct = Cache::remember('popular_vip_product_'.$user->id, now()->addHours(6), function() {
        return VipProduct::withCount(['investments as investors_count' => function($query) {
                $query->where('expires_at', '>', now());
            }])
            ->active()
            ->orderBy('investors_count', 'desc')
            ->orderBy('is_featured', 'desc')
            ->first();
    });

    if ($popularProduct) {
        $dailyReturn = ($popularProduct->earnings_per_user / $popularProduct->price) * 100;
        
        return [
            'product' => $popularProduct,
            'product_id' => $popularProduct->id,
            'name' => $popularProduct->investment_tool,
            'price' => $popularProduct->price,
            'daily_earnings' => $popularProduct->earnings_per_user,
            'daily_return' => $dailyReturn,
            'investors_count' => $popularProduct->investors_count,
            'user_code' => 'VIP-'.$user->id,
            'is_available' => $popularProduct->isAvailableFor($user)
        ];
    }

    return [
        'product' => null,
        'name' => 'VIP প্যাকেজ লোড হচ্ছে...',
        'price' => 0,
        'daily_return' => 0,
        'user_code' => 'VIP-'.$user->id,
        'is_available' => false
    ];
}

    private function getRiskAvoidance($user)
    {
        // Calculate based on user's investment diversity
        $investmentTypes = UserVipInvestment::where('user_id', $user->id)
            ->select('type')
            ->distinct()
            ->count();
            
        $totalInvestments = UserVipInvestment::where('user_id', $user->id)->count();
        
        if ($totalInvestments > 0) {
            $diversityScore = min(($investmentTypes / $totalInvestments) * 100, 100);
            return round(max($diversityScore, 30)); // Minimum 30% score
        }
        
        return 30; // Default score for new users
    }
}