<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\UserVipInvestment;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

use Illuminate\Support\Facades\Artisan; 
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class VipController extends Controller
{
public function processEarningsManually(Request $request)
{
    try {
        $investments = UserVipInvestment::with(['user'])
            ->where('is_active', true)
            ->where('expires_at', '>', now())
            ->where(function($query) {
                $query->whereNull('last_payout_at')
                      ->orWhere('last_payout_at', '<=', now()->subHours(24));
            })
            ->get();

        $processed = 0;
        $skipped = 0;
        
        foreach ($investments as $investment) {
            try {
                DB::transaction(function () use ($investment, &$processed, &$skipped) {
                    // ইউজার আছে কিনা চেক করুন
                    if (!$investment->user) {
                        Log::error("User not found for investment ID: {$investment->id}");
                        $skipped++;
                        return;
                    }

                    // ব্যালেন্স আপডেট করুন
                    $investment->user->increment('balance', $investment->daily_earnings);
                    
                    // ট্রানজেকশন রেকর্ড তৈরি করুন
                    Transaction::create([
                        'user_id' => $investment->user_id,
                        'amount' => $investment->daily_earnings,
                        'type' => 'vip_earnings',
                        'status' => 'completed',
                        'notes' => 'দৈনিক ভিআইপি আয়'
                    ]);
                    
                    // শেষ পেমেন্ট টাইম আপডেট করুন
                    $investment->update(['last_payout_at' => now()]);
                    $processed++;
                });
            } catch (\Exception $e) {
                Log::error("Error processing investment ID: {$investment->id} - " . $e->getMessage());
                $skipped++;
                continue;
            }
        }
        
        return response()->json([
            'success' => true,
            'processed' => $processed,
            'skipped' => $skipped,
            'message' => "প্রসেস সম্পূর্ণ! সফল: {$processed} টি, বাদ পড়েছে: {$skipped} টি"
        ]);
        
    } catch (\Exception $e) {
        Log::error("VIP Earnings Processing Error: " . $e->getMessage());
        return response()->json([
            'success' => false,
            'processed' => 0,
            'message' => 'ত্রুটি: ' . $e->getMessage()
        ], 500);
    }
}
public function processSingleEarning($id)
{
    try {
        $investment = UserVipInvestment::with('user')->findOrFail($id);
        
        // ভ্যালিডেশন চেক
        if (!$investment->is_active) {
            return back()->with('error', 'এই ইনভেস্টমেন্টটি একটিভ নেই');
        }
        
        if ($investment->expires_at <= now()) {
            return back()->with('error', 'এই ইনভেস্টমেন্টটির মেয়াদ শেষ');
        }
        
        if ($investment->last_payout_at && $investment->last_payout_at > now()->subHours(24)) {
            return back()->with('error', 'এই ইনভেস্টমেন্টের জন্য ২৪ ঘন্টা পূর্ণ হয়নি');
        }
        
        DB::beginTransaction();
        
        // ১. ইউজারের ব্যালেন্স আপডেট
        $newBalance = $investment->user->balance + $investment->daily_earnings;
        $investment->user->update(['balance' => $newBalance]);
        
        // ২. ট্রানজেকশন রেকর্ড
        Transaction::create([
            'user_id' => $investment->user_id,
            'amount' => $investment->daily_earnings,
            'type' => 'vip_earnings',
            'status' => 'completed',
            'balance' => $newBalance,
            'notes' => 'ভিআইপি আয় - ' . $investment->product->investment_tool
        ]);
        
        // ৩. শেষ পেমেন্ট টাইম আপডেট
        $investment->update(['last_payout_at' => now()]);
        
        DB::commit();
        
        return back()->with('success', 
            'আয় সফলভাবে প্রসেস করা হয়েছে! ' .
            number_format($investment->daily_earnings, 2) . ' ৳ যোগ করা হয়েছে ' .
            $investment->user->name . ' এর অ্যাকাউন্টে'
        );
            
    } catch (\Exception $e) {
        DB::rollBack();
        return back()->with('error', 'ত্রুটি: ' . $e->getMessage());
    }
}
    public function pendingEarnings()
    {
        $pendingInvestments = UserVipInvestment::with(['user', 'product'])
            ->where('is_active', true)
            ->where('expires_at', '>', now())
            ->where(function($query) {
                $query->whereNull('last_payout_at')
                      ->orWhere('last_payout_at', '<=', now()->subHours(24));
            })
            ->get();
            
        return view('admin.vip.pending_earnings', compact('pendingInvestments'));
    }
}