<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PaymentMethod;
use Illuminate\Http\Request;

class AdminPaymentMethodController extends Controller
{
    public function index()
    {
        $methods = PaymentMethod::orderBy('sort_order')->get();
        return view('admin.payment-methods.index', compact('methods'));
    }

    public function create()
    {
        return view('admin.payment-methods.create');
    }

    public function store(Request $request)
    {
        
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
              'description' => 'required|string|max:500',
            'account_number' => 'required|string|max:255',
            'icon' => 'nullable|image|mimes:png,jpg,jpeg|max:2048',

            'is_active' => 'sometimes|accepted'
        ]);

        try {
            $method = new PaymentMethod();
            $method->name = $validated['name'];
            $method->description = $validated['description'];
            $method->account_number = $validated['account_number'];
            $method->is_active = $request->has('is_active');
            $method->sort_order = PaymentMethod::count() + 1;

            if ($request->hasFile('icon')) {
                $file = $request->file('icon');
                $filename = 'pm-'.time().'.'.$file->getClientOriginalExtension();
                
                // Store in public/images/payment-methods
                $path = $file->move(public_path('images/payment-methods'), $filename);
                
                // Save relative path to database
                $method->icon = 'images/payment-methods/'.$filename;
            }

            $method->save();
            
            return redirect()->route('admin.payment-methods')
                   ->with('success', 'Payment method added successfully!');

        } catch (\Exception $e) {
            return back()->with('error', 'Error: '.$e->getMessage())
                         ->withInput();
        }
    }

    public function edit(PaymentMethod $method)
    {
        return view('admin.payment-methods.edit', compact('method'));
    }

    public function update(Request $request, PaymentMethod $method)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'account_number' => 'required|string|max:255',
            'description' => 'required|string|max:500',
            'icon' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        $data = $request->only(['name', 'description','account_number', 'is_active']);

        // Handle icon removal
        if ($request->has('remove_icon') && $request->remove_icon == '1') {
            if ($method->icon) {
                // Delete old icon file
                if (file_exists(public_path($method->icon))) {
                    unlink(public_path($method->icon));
                }
                $data['icon'] = null;
            }
        }

        // Handle new icon upload
        if ($request->hasFile('icon')) {
            // Delete old icon if exists
            if ($method->icon && file_exists(public_path($method->icon))) {
                unlink(public_path($method->icon));
            }

            $icon = $request->file('icon');
            $filename = 'payment-method-'.time().'.'.$icon->getClientOriginalExtension();
            $path = 'images/payment-methods/';
            
            // Create directory if not exists
            if (!file_exists(public_path($path))) {
                mkdir(public_path($path), 0777, true);
            }

            $icon->move(public_path($path), $filename);
            $data['icon'] = $path.$filename;
        }

        $method->update($data);

        return redirect()->route('admin.payment-methods')
                        ->with('success', 'Payment method updated successfully');
    }

    public function destroy(PaymentMethod $method)
{
    try {
        // Delete icon file if exists
        if ($method->icon && file_exists(public_path($method->icon))) {
            unlink(public_path($method->icon));
        }

        $method->delete();

        return response()->json([
            'success' => true,
            'message' => 'Payment method deleted successfully'
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error deleting payment method: ' . $e->getMessage()
        ], 500);
    }
}

    public function updateStatus(Request $request, PaymentMethod $method)
    {
        $method->is_active = $request->status == 'active';
        $method->save();
        return response()->json(['success' => true]);
    }

    public function updateOrder(Request $request)
    {
        foreach ($request->order as $order => $id) {
            PaymentMethod::where('id', $id)->update(['sort_order' => $order]);
        }
        return response()->json(['success' => true]);
    }
}