<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\UserVipInvestment;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\Transaction; // এই লাইনটি যোগ করুন
class ProcessVipEarnings extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'vip:process-earnings';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Process daily earnings for VIP investments';

    /**
     * Execute the console command.
     */
  public function handle()
{
    $this->info('Starting VIP earnings processing...');
    
 $investments = UserVipInvestment::with(['user'])
        ->active()
        ->where(function($query) {
            $query->whereNull('last_payout_at')
                  ->orWhere('last_payout_at', '<=', now()->subHours(24));
        })
        ->get();

    $this->info('Found '.$investments->count().' active investments to process');

    foreach ($investments as $investment) {
        if (!$investment->user) {
            $this->error("User not found for investment ID: {$investment->id}");
            continue;
        }

        DB::beginTransaction();
        
        try {
            $this->info("Processing investment ID: {$investment->id} for user ID: {$investment->user_id}");

            // ব্যালেন্স ইনক্রিমেন্ট
            $investment->user->increment('balance', $investment->daily_earnings);
            
            // লেনদেন রেকর্ড
            Transaction::create([
                'user_id' => $investment->user_id,
                'amount' => $investment->daily_earnings,
                'type' => Transaction::TYPE_VIP_EARNINGS,
                'status' => Transaction::STATUS_COMPLETED,
                'method' => 'vip_investment',
                'transaction_id' => 'VIP-EARN-'.uniqid(),
                'notes' => "Daily earnings from VIP investment"
            ]);
            
            // শেষ পেমেন্ট টাইম আপডেট (বর্তমান সময়)
            $investment->update(['last_payout_at' => now()]);
            
            DB::commit();
            
            $this->info("Successfully added {$investment->daily_earnings} to user {$investment->user_id}");

        } catch (\Exception $e) {
            DB::rollBack();
            $this->error("Failed to process investment {$investment->id}: ".$e->getMessage());
            Log::error("VIP Earnings Error: ".$e->getMessage(), [
                'investment' => $investment->toArray(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
    
    $this->info('VIP earnings processing completed!');
}
}